
#include <linux/moduleparam.h>
#include <asm/delay.h>
#include "iprq.h"
#include "iprq_common.h"

static iprq_handler_t iprq_handler[IPRQ_REQUEST_MAXNUM];


void send_iprq(int osid)
{
	IRQ_REQUEST(GET_IPRQ_IRQ(osid));
}

static int __pop_proc_request(struct iprq_ctlblk_t *ctlblk,
			      struct proc_request_t *preq)
{
	struct proc_request_t *p;
	/* no one decreases remain, and no one changes first, excluding self */
#ifdef IPRQ_ASSERT
	if (ctlblk->remain <= 0) IPRQ_ABORT();
#endif
	p = &ctlblk->buf[ctlblk->first];

	MEMCPY(preq, p, sizeof(struct proc_request_t));

	MB();

	ctlblk->first = (ctlblk->first >= IPRQ_REQUEST_BUFSIZE - 1) ?
			0 : ctlblk->first + 1;
	ctlblk->remain--;
#ifdef IPRQ_ASSERT
	if ((ctlblk->last == ctlblk->first) ^
	    (ctlblk->remain % IPRQ_REQUEST_BUFSIZE == 0)) {
		IPRQ_ABORT();
	}
#endif
	return 0;
}

static int __push_proc_request(struct iprq_ctlblk_t *ctlblk,
			       struct proc_request_t *preq)
{
	struct proc_request_t *p;
	unsigned long flags;

	flags = LOCK_CLI_SAVE(&ctlblk->lock);

	if (ctlblk->remain >= IPRQ_REQUEST_BUFSIZE) {
		UNLOCK_IRQ_RESTORE(&ctlblk->lock, flags);
		return E_IPRQ_BUSY;
	}
	p = &ctlblk->buf[ctlblk->last];

	MEMCPY(p, preq, sizeof(struct proc_request_t));

	MB();

	ctlblk->last = (ctlblk->last >= IPRQ_REQUEST_BUFSIZE - 1) ?
		       0 : ctlblk->last + 1;
	ctlblk->remain++;
#ifdef IPRQ_ASSERT
	if ((ctlblk->last == ctlblk->first) ^
	    (ctlblk->remain % IPRQ_REQUEST_BUFSIZE == 0)) {
		IPRQ_ABORT();
	}
#endif
	UNLOCK_IRQ_RESTORE(&ctlblk->lock, flags);
	return 0;
}

void do_proc_request(void)
{
	struct proc_request_t preq;
	int request, retval;
	iprq_handler_t handler;
	unsigned long flags;
	void *msg;
	struct iprq_ctlblk_t *self_ctlblk = &iprq_ctlblk[SELF_OSID()];
	int write_offset;

	if (!is_iprq_initialized()) {
		IPRQ_ABORT();
	}

	flags = LOCK_CLI_SAVE(&iprq_printcb->read_lock);
	write_offset = iprq_printcb->write_offset;
	while (iprq_printcb->read_offset != write_offset) {
		char c = iprq_printcb->buf[iprq_printcb->read_offset++];
		if (iprq_printcb->read_offset == IPRQ_PRINT_BUFSIZE) {
			iprq_printcb->read_offset = 0;
		}
		UNLOCK_IRQ_RESTORE(&iprq_printcb->read_lock, flags);
		printk ("%c", c);
		flags = LOCK_CLI_SAVE(&iprq_printcb->read_lock);
		write_offset = iprq_printcb->write_offset;
	}
	UNLOCK_IRQ_RESTORE(&iprq_printcb->read_lock, flags);

	flags = LOCK_CLI_SAVE(&self_ctlblk->lock);
	while (iprq_ctlblk[SELF_OSID()].remain) {
		__pop_proc_request(&iprq_ctlblk[SELF_OSID()], &preq);
//		UNLOCK_IRQ_RESTORE(&self_ctlblk->lock, flags);
		request = preq.request;

		handler = (request >= 0 && request < IPRQ_REQUEST_MAXNUM) ?
			  iprq_handler[request] : NULL;
#ifdef IPRQ_DEBUG
		if (handler == NULL) IPRQ_ABORT();
#endif
		if (preq.size > sizeof(unsigned int)) {
			msg = iprqX_attach_mem(preq.arg);
#ifdef IPRQ_ASSERT
			if (!msg) {
				IPRQ_ABORT();
			}
#endif
		} else {
			msg = &preq.arg;
		}
		retval = (handler) ? handler(request, msg) : 0;

		if (preq.size > sizeof(unsigned int)) {
			iprqX_detach_mem(msg);
			iprqX_free_mem(preq.arg);
		}

//		LOCK_CLI_SAVE(&self_ctlblk->lock);
		if (preq.sync == __PREQ_SYNC) {
			iprq_ctlblk[preq.sender].retval = retval;
			MB();
			iprq_ctlblk[preq.sender].finish = 1;
		}
	}
	UNLOCK_IRQ_RESTORE(&self_ctlblk->lock, flags);
}

static int __set_proc_request_message(struct proc_request_t *ppreq, void *msg)
{
	unsigned int size = ppreq->size;
	int memid;
	void *p;

	if (size > 0 && !msg) return E_IPRQ_INVAL;

	if (size > sizeof(unsigned int)) {
		memid = iprqX_alloc_mem(size);
		if (memid < 0) {
			return E_IPRQ_NOMEM;
		}
		p = iprqX_attach_mem(memid);
#ifdef IPRQ_ASSERT
		if (!p) {
			IPRQ_ABORT();
		}
#endif
		MEMCPY(p, msg, size);
		iprqX_detach_mem(p);
		ppreq->arg = memid;
	} else if (size > 0) {
		ppreq->arg = *(unsigned int*)msg;
	} else {
		ppreq->arg = 0;
	}
	return 0;
}

static int __unset_proc_request_message(struct proc_request_t *ppreq)
{
	unsigned int size = ppreq->size;
	int err = 0;
	if (size > sizeof(unsigned int)) {
		err = iprqX_free_mem(ppreq->arg);
#ifdef IPRQ_ASSERT
		if (err) {
			IPRQ_ABORT();
		}
#endif
	}
	return err;
}

static int __iprqX_send(int osid, int req, void *msg, unsigned int size)
{
	struct proc_request_t preq =
		{req, SELF_OSID(), __PREQ_ASYNC, size, 0};
	int err;

	err = __set_proc_request_message(&preq, msg);
	if (err) {
		return err;
	}
	err = __push_proc_request(&iprq_ctlblk[osid], &preq);
	if (err) {
		__unset_proc_request_message(&preq);
	}
	return err;
}

int iprqX_send(int osid, int req, void *msg, unsigned int size)
{
#ifdef IPRQ_UNSUPPORTED
	if (osid == SELF_OSID()) {
		IPRQ_ABORT();
	}
#endif
	if (!is_iprq_initialized()) {
		IPRQ_ABORT();
	}

	if (req < 0 || req >= IPRQ_REQUEST_MAXNUM) {
		return E_IPRQ_INVAL;
	}

	return __iprqX_send(osid, req, msg, size);
}

static volatile int __iprq_dsp_stopped = 0;

static void __iprq_stop_dsp(void)
{
	unsigned int timeout = IPRQ_POLL_TIMEOUT;

	NMI_ENABLE(IPRQ_L_TO_R_NMI);
	NMI_REQUEST(IPRQ_L_TO_R_NMI);
	do {
		if (!IS_IRQ_DETECTED(IPRQ_L_TO_R_NMI) || __iprq_dsp_stopped) {
			break;
		}
		MB();
		udelay(1);
	} while (--timeout);
	if (!timeout) {
		printk(KERN_ERR "IPRQ: ERROR: %s: NMI is not handled\n", __func__);
		return;
	}
	__iprq_dsp_stopped = 1;
}

static void __iprq_dump_dsp_registers(void)
{
	unsigned int *ptr;
	unsigned long base;

	if (!__iprq_dsp_stopped) {
		__iprq_stop_dsp();
	}

	ptr = ioremap_nocache(DDR_DSP_STATUS_BASE, DDR_DSP_STATUS_SIZE);
	if (!ptr) {
		printk(KERN_ERR "IPRQ: ERROR: %s: ioremap error\n", __func__);
		return;
	}
	base = (unsigned long)ptr;
	printk("IPRQ: dump DSP registers >>>>>\n");
	printk("pc : %08x  lr : %08x  psr: %08x\n",
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_PC)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_LR)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_PSR)));
	printk("esb: %08x\n",
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_ESB)));
	printk("r0 : %08x  r1 : %08x  r2 : %08x\n",
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R0)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R1)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R2)));
	printk("r3 : %08x  r4 : %08x  r5 : %08x\n",
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R3)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R4)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R5)));
	printk("r6 : %08x  r7 : %08x  r8 : %08x\n",
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R6)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R7)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R8)));
	printk("r9 : %08x  r10: %08x  r11: %08x\n",
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R9)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R10)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R11)));
	printk("r12: %08x  r13: %08x  r14: %08x\n",
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R12)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R13)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R14)));
	printk("r15: %08x  r16: %08x  r17: %08x\n",
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R15)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R16)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R17)));
	printk("r18: %08x  r19: %08x  r20: %08x\n",
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R18)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R19)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R20)));
	printk("r21: %08x  r22: %08x  r23: %08x\n",
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R21)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R22)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R23)));
	printk("r24: %08x  r25: %08x  r26: %08x\n",
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R24)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R25)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R26)));
	printk("r27: %08x  r28: %08x  r29: %08x\n",
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R27)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R28)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R29)));
	printk("r30: %08x  r31: %08x\n",
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R30)),
		*((volatile unsigned int *)(base + DDR_DSP_STATUS_OFFSET_R31)));
	printk("IPRQ: dump DSP registers <<<<<\n");
	iounmap(ptr);
}

static unsigned int iprq_timeout = IPRQ_POLL_TIMEOUT;
module_param(iprq_timeout, uint, 0644);

static DEFINE_LOCAL_LOCK(iprq_send_sync_lock);

static int __iprqX_send_sync(int osid, int req, void *msg,
		      unsigned int size, int *ret)
{
	struct proc_request_t preq =
		{req, SELF_OSID(), __PREQ_SYNC, size, 0};
	struct iprq_ctlblk_t *self_ctlblk = &iprq_ctlblk[SELF_OSID()];
	int err;
	unsigned int timeout = iprq_timeout;
	__DSP_INIT(state);

	LOCAL_LOCK(&iprq_send_sync_lock, state);

	self_ctlblk->finish = 0;
	MB();

	err = __set_proc_request_message(&preq, msg);
	if (err) {
		goto out;
	}
	err = __push_proc_request(&iprq_ctlblk[osid], &preq);
	if (err) {
		__unset_proc_request_message(&preq);
		goto out;
	}
	send_iprq(osid);

	if (timeout) {
		do {
			if (self_ctlblk->finish) {
				break;
			}
			MB();
			udelay(1);
		} while (--timeout);
		if (!timeout) {
			printk(KERN_ERR "IPRQ: ERROR: %s timeout.\n", __func__);
			if (IS_IRQ_DETECTED(GET_IPRQ_IRQ(osid))) {
				printk(KERN_ERR "IPRQ: %s: IRQ is NOT cleared.\n",
					__func__);
			}
			else {
				printk(KERN_ERR "IPRQ: %s: IRQ had been cleared.\n",
					__func__);
			}
			__iprq_dump_dsp_registers();
#ifdef IPRQ_TIMEOUT_SIGNAL
			printk(KERN_ERR "IPRQ: %s: send signal %d to %s(tid=%d).\n",
				__func__,
				IPRQ_TIMEOUT_SIGNAL, current->comm, current->pid);
			force_sig(IPRQ_TIMEOUT_SIGNAL, current);
#else
			panic("IPRQ: DSP not responding");	
#endif /* IPRQ_TIMEOUT_SIGNAL */
		}
	}
	else {
		while (!self_ctlblk->finish) {
			MB();
		}
	}

	if (ret) {
		*ret = self_ctlblk->retval;
	}

out:
	LOCAL_UNLOCK(&iprq_send_sync_lock, state);
	return err;
}

int iprqX_send_sync(int osid, int req, void *msg,
		      unsigned int size, int *ret)
{
#ifdef IPRQ_UNSUPPORTED
	if (osid == SELF_OSID()) {
		IPRQ_ABORT();
	}
#endif
	if (!is_iprq_initialized()) {
		IPRQ_ABORT();
	}

	if (req < 0 || req >= IPRQ_REQUEST_MAXNUM) {
		return E_IPRQ_INVAL;
	}

	return __iprqX_send_sync(osid, req, msg, size ,ret);
}

 /* iprqX_send_sync_timeout is not implemented yet. */
int iprqX_send_sync_timeout(int osid, int req, void *msg,
		      unsigned int size, int *ret, unsigned int usec)
{
#ifdef IPRQ_UNSUPPORTED
	IPRQ_ABORT();
#endif
	return 0;
}

static int __iprqX_entry(int request, iprq_handler_t handler)
{
	iprq_handler[request] = handler;
	return 0;
}

int iprqX_entry(int request, iprq_handler_t handler)
{
	if (request < 0 || request >= IPRQ_REQUEST_MAXNUM) {
#ifdef IPRQ_DEBUG
		IPRQ_ABORT();
#endif
		return E_IPRQ_INVAL;
	}
	return __iprqX_entry(request, handler);
}

int iprq_is_initialized(void)
{
	struct iprq_status_t *status = iprq_status;
	unsigned long flags;
	int ret;

	flags = LOCK_CLI_SAVE(&status->lock);
	ret = __is_all_os_initialized(status);
	UNLOCK_IRQ_RESTORE(&status->lock, flags);

	return ret;
}
